<?php
/**
 * Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2023 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * http://cloudlinux.com/docs/LICENSE.TXT
 */

namespace Modules_AccelerateWp_ApiRpc;

use pm_ApiCli;
use pm_Exception;

class CloudlinuxAwpAdminCommand
{
    /**
     * @var string[]
     */
    private array $commands = [
        'set-suite' => 'setSuite',
        'set-whmcs' => 'setWhmcs',
        'set-whmcs-stat' => 'setWhmcsStat',
        'get-whmcs-config' => 'getWhmcsConfig',
        'restore-whmcs-backup' => 'restoreWhmcsBackup',
        'set-options' => 'setOptions',
    ];

    /**
     * @param array $args
     *
     * @return array
     * @throws pm_Exception
     */
    public function run($args)
    {
        if (!is_array($args)) {
            $args = [];
        }

        $command = (array_key_exists('command', $args) && !is_array($args['command'])) ? $args['command'] : '';
        $commands = array_keys($this->commands);

        if (!in_array($command, $commands)) {
            throw new pm_Exception('Command not found, available commands: ' . implode(', ', $commands), 404, null);
        }

        $command_method = $this->commands[$command];

        $params = $this->$command_method($args);

        return $this->apiCliCallSbin($params);
    }

    /**
     * @param array $args
     * @return array
     * @throws pm_Exception
     */
    protected function setSuite(array $args)
    {
        $visible_for_all = array_key_exists('visible-for-all', $args) ? true : null;
        $allowed_for_all = array_key_exists('allowed-for-all', $args) ? true : null;
        $disallowed_for_all = array_key_exists('disallowed-for-all', $args) ? true : null;

        if ($visible_for_all === true || $allowed_for_all === true || $disallowed_for_all === true) {
            return $this->setSuiteForAll($args);
        }

        $suites = (array_key_exists('suites', $args) && !is_array($args['suites'])) ? $this->validateArgument($args['suites']) : '';
        $users = (array_key_exists('users', $args) && !is_array($args['users'])) ? $this->validateArgument($args['users']) : '';
        $domains = (array_key_exists('domains', $args) && !is_array($args['domains'])) ? $this->validateArgument($args['domains']) : '';
        $default = array_key_exists('default', $args) ? true : null;
        $visible = array_key_exists('visible', $args) ? true : null;
        $allowed = array_key_exists('allowed', $args) ? true : null;
        $disallowed = array_key_exists('disallowed', $args) ? true : null;
        $attrs = (array_key_exists('attrs', $args) && is_string($args['attrs'])) ? $this->validateAttrs($args['attrs']) : [];

        if (empty($suites)) {
            throw new pm_Exception('Provide a comma separated list of suites', 404, null);
        }

        if (empty($users)) {
            throw new pm_Exception('Provide a comma separated list of users', 404, null);
        }

        if (empty($domains)) {
            throw new pm_Exception('Provide a comma separated list of domains', 404, null);
        }

        if (is_null($default) && is_null($visible) && is_null($allowed) && is_null($disallowed)) {
            throw new pm_Exception(
                'Attribute default, visible, allowed or disallowed is not defined',
                404,
                null
            );
        }

        $usernames = $this->findUsernames($users, $domains);

        $params = [
            1 => 'set-suite',
            2 => $suites,
            3 => $usernames,
            4 => 'BILLING_OVERRIDE',
        ];

        if ($default === true) {
            $params[5] = '--default';
        } elseif ($visible === true) {
            $params[5] = '--visible';
        } elseif ($allowed === true) {
            $params[5] = '--allowed';
        } elseif ($disallowed === true) {
            $params[5] = '--disallowed';
        }

        $params[6] = json_encode($attrs);

        return $params;
    }

    /**
     * @param array $args
     * @return array
     * @throws pm_Exception
     */
    protected function setSuiteForAll(array $args)
    {
        $suites = (array_key_exists('suites', $args) && !is_array($args['suites'])) ? $this->validateArgument($args['suites']) : '';
        $visible_for_all = array_key_exists('visible-for-all', $args) ? true : null;
        $allowed_for_all = array_key_exists('allowed-for-all', $args) ? true : null;
        $disallowed_for_all = array_key_exists('disallowed-for-all', $args) ? true : null;
        $attrs = (array_key_exists('attrs', $args) && is_string($args['attrs'])) ? $this->validateAttrs($args['attrs']) : [];

        if (empty($suites)) {
            throw new pm_Exception('Provide a comma separated list of suites', 404, null);
        }

        if (is_null($visible_for_all) && is_null($allowed_for_all) && is_null($disallowed_for_all)) {
            throw new pm_Exception(
                'Attribute visible-for-all, allowed-for-all nor disallowed-for-all is not defined',
                404,
                null
            );
        }

        $params = [
            1 => 'set-suite-for-all',
            2 => $suites,
            3 => 'BILLING_OVERRIDE',
        ];

        if ($visible_for_all === true) {
            $params[4] = '--visible-for-all';
        } elseif ($allowed_for_all === true) {
            $params[4] = '--allowed-for-all';
        } elseif ($disallowed_for_all === true) {
            $params[4] = '--disallowed-for-all';
        }

        $params[5] = json_encode($attrs);

        return $params;
    }

    /**
     * @param array $args
     * @return array
     * @throws pm_Exception
     */
    protected function setWhmcs(array $args)
    {
        $suites = (array_key_exists('suites', $args) && !is_array($args['suites'])) ? $this->validateArgument($args['suites']) : '';
        $backup = array_key_exists('backup', $args) ? true : null;
        $setup = array_key_exists('setup', $args) ? true : null;

        if (is_null($setup)) {
            throw new pm_Exception('Attribute setup is not defined', 404, null);
        }

        $params = [
            1 => 'set-whmcs',
            2 => '--setup',
            3 => '',
            4 => '',
        ];

        if ($backup) {
            $params['3'] = '--backup';
        }

        if (!empty($suites)) {
            $params['4'] = '--suites=' . $suites;
        }

        return $params;
    }

    /**
     * @param array $args
     * @return array
     * @throws pm_Exception
     */
    protected function setWhmcsStat(array $args)
    {
        $json = (array_key_exists('json', $args) && !is_array($args['json'])) ? $args['json'] : '';
        $stat = json_decode($json, true);

        if (!is_array($stat) or empty($stat)) {
            throw new pm_Exception('Attribute json is not valid json', 404, null);
        }

        foreach ($stat as $key => $val) {
            unset($stat[$key]);
            $key = $this->validateArgument($key);
            $stat[$key] = (int)$val;
        }

        return [
            1 => 'set-whmcs-stat',
            2 => json_encode($stat),
        ];
    }

    /**
     * @return array
     * @throws pm_Exception
     */
    protected function getWhmcsConfig()
    {
        return [
            1 => 'get-whmcs-config',
        ];
    }

    /**
     * @return array
     * @throws pm_Exception
     */
    protected function restoreWhmcsBackup()
    {
        return [
            1 => 'restore-whmcs-backup',
        ];
    }

    /**
     * @param array $args
     * @return array
     * @throws pm_Exception
     */
    protected function setOptions(array $args)
    {
        $upgrade_url = (array_key_exists('upgrade-url', $args) && !is_array($args['upgrade-url'])) ? $args['upgrade-url'] : '';

        if (filter_var($upgrade_url, FILTER_VALIDATE_URL) === false) {
            throw new pm_Exception('Provide upgrade url', 404, null);
        }

        return [
            1 => 'set-options',
            2 => $upgrade_url,
        ];
    }

    /**
     * @param string $users
     * @param string $domains
     *
     * @return string
     * @throws pm_Exception
     */
    protected function findUsernames(string $users, string $domains)
    {
        $result = [];

        $users = explode(',', $users);
        $users = array_map('trim', $users);
        $domains = explode(',', $domains);
        $domains = array_map('trim', $domains);

        foreach ($domains as $key => $domain) {
            $user = array_key_exists($key, $users) ? $users[$key] : '';

            $subscription_username = $this->findSubscriptionUsername($domain);

            if (empty($user)) {
                throw new pm_Exception('Not received user for domain ' . $domain, 404, null);
            } elseif (empty($subscription_username)) {
                throw new pm_Exception('Subscription username for domain ' . $domain . ' not found', 404, null);
            } elseif ($subscription_username !== $user) {
                throw new pm_Exception('Subscription username for domain ' . $domain . ' not equal to user ' . $user, 404, null);
            } else {
                $result[] = $subscription_username;
            }
        }

        return implode(',', $result);
    }

    /**
     * @param string $domain
     *
     * @return string
     */
    protected function findSubscriptionUsername(string $domain)
    {
        $ftp_login = '';

        $data = $this->apiCliCall(['-i', $domain]);
        $stdout = $data['stdout'];

        preg_match('/^FTP Login:(.*)/im', $stdout, $m);
        if (array_key_exists(1, $m)) {
            $ftp_login = trim($m[1]);
        }

        return $ftp_login;
    }

    /**
     * @param string $str
     *
     * @return string
     */
    protected function validateArgument($str)
    {
        return (string)preg_replace('/[^a-z\d\s.,_-]/i', '', $str);
    }

    /**
     * @param string $json
     *
     * @return array
     */
    protected function validateAttrs($json)
    {
        $attrs = json_decode($json, true);
        $validated = [];

        if (is_array($attrs) and array_key_exists('traffic_limit', $attrs)) {
            $validated['traffic_limit'] = $this->validateArgument($attrs['traffic_limit']);
        }

        return $validated;
    }

    /**
     * @param array $args
     * @return array
     */
    protected function apiCliCallSbin(array $args)
    {
        return pm_ApiCli::callSbin('cloudlinux-awp-admin', $args, pm_ApiCli::RESULT_FULL);
    }

    /**
     * @param array $args
     * @return array
     */
    protected function apiCliCall(array $args)
    {
        return pm_ApiCli::call('site', $args, pm_ApiCli::RESULT_FULL);
    }
}